/*
*********************************************************************************************************
*                                               uC/OS-II
*                                         The Real-Time Kernel
*
*                        (c) Copyright 1992-2002, Jean J. Labrosse, Weston, FL
*                                          All Rights Reserved
*
*********************************************************************************************************
*
* Port to Mitsubishi M16C/60 Series
*
* (c) Florian Stassen
* started 2003-01-18
*
*********************************************************************************************************
*/

#define  OS_CPU_GLOBALS
#include "includes.h"

/*$PAGE*/
/*
*********************************************************************************************************
*                                        INITIALIZE A TASK'S STACK
*
* Description: This function is called by either OSTaskCreate() or OSTaskCreateExt() to initialize the
*              stack frame of the task being created.  This function is highly processor specific.
*
* Arguments  : task          is a pointer to the task code
*
*              pdata         is a pointer to a user supplied data area that will be passed to the task
*                            when the task first executes.
*
*              ptos          is a pointer to the top of stack.  It is assumed that 'ptos' points to
*                            a 'free' entry on the task stack.  If OS_STK_GROWTH is set to 1 then
*                            'ptos' will contain the HIGHEST valid address of the stack.  Similarly, if
*                            OS_STK_GROWTH is set to 0, the 'ptos' will contains the LOWEST valid address
*                            of the stack.
*
*              opt           specifies options that can be used to alter the behavior of OSTaskStkInit().
*                            (see uCOS_II.H for OS_TASK_OPT_???).
*
* Returns    : Always returns the location of the new top-of-stack' once the processor registers have
*              been placed on the stack in the proper order.
*
* Note(s)    : 1) Interrupts are enabled when your task starts executing. You can change this by clearing the
*                 I bit in the FLG register.  In this case, interrupts would be disabled upon task startup.  The
*                 application code would be responsible for enabling interrupts at the beginning of the task
*                 code.  You will need to modify OSTaskIdle() and OSTaskStat() so that they enable
*                 interrupts.  Failure to do this will make your system crash!
*              2) Task stack frame looks as follows.  The stack width is assumed to be 16 bits wide:
*
*                       SP -->  R0              (Low memory)
*                               R1
*                               R2
*                               R3
*                               A0
*                               A1
*                               SB
*                               FB
*                               PC
*                     ptos -->  PC/FLG          (High memory)
*
*********************************************************************************************************
*/

OS_STK  *OSTaskStkInit (void (*task)(void *pd), void *pdata, OS_STK *ptos, INT16U opt)
{
  #define FLAG  0x00C0   /* Contents of FLG register ie. IPL=0, USP selected, Interrupts enabled */

  INT8U Temp1;
  INT8U Temp2;
  INT16U Temp3;

  INT16U * stk16;

  //stk16 = (INT16U *)ptos;         /* Load top of stack for this task   */
  // Better stack alignment with optimization "stack_frame_align" with NC30WA Compiler
  // see ReadMe.txt
  Temp3 = (INT16U)ptos - 1;
  stk16 = (INT16U *)Temp3;          /* Load top of stack for this task   */

  Temp1 = (INT8U) (FLAG >> 8);  /* get the MSB of FLG register */
  Temp1 &= 0xF0;                /* Only keep the MS 4 bits of FLG in Temp1 */
                                /* The other 4 should be the MS 4 bits of PC */

  Temp2 = (INT32U)task >> 16;   /* get the 3rd LSB in task address */
  Temp2 &= 0x0F;                /* keep the Low nibble */

  Temp1 |= Temp2;               /* High nibble of Temp1 now contains MS 4bits of FLG register */
                                /* Low nibble of Temp1 now contains MS 4 bits of task address */

  Temp3 = FLAG & 0x00FF;        // LSB of Flags

  Temp3 |= (INT16U)Temp1 << 8;


  *stk16 = Temp3;               /* push PC/FLG to stack */

  stk16--;
  *stk16 = (INT32U)task;        /* push 16 bit LSB in Task address */

  stk16 -= 6;                   /* simulate pushing FB, SB, A1, A0, R3, R2 ie 6 reg 2 bytes each --> 12 */

  stk16--;
  *stk16 =  (INT16U)pdata;      /* save pdata in R1 since nc30 compiler uses R1 to pass pdata */

  stk16--;                      /* simulate pushing R0 */

  return ((OS_STK *)stk16);
}

